'use client';

import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from '@/components/ui/card';
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert';
import { Button } from '@/components/ui/button';
import { ShoppingCart, Search, FileDown, ArrowLeft } from 'lucide-react';
import { Input } from '@/components/ui/input';
import { useCollection, useFirestore, useMemoFirebase } from '@/firebase';
import { collection, doc } from 'firebase/firestore';
import { useParams, useRouter } from 'next/navigation';
import type { Order, User } from '@/lib/data';
import { useDoc } from '@/firebase/firestore/use-doc';
import { Skeleton } from '@/components/ui/skeleton';
import { OrderCard } from '@/components/order-card';

export default function UserOrdersPage() {
  const params = useParams();
  const router = useRouter();
  const userId = params.userId as string;
  const firestore = useFirestore();

  const userDocRef = useMemoFirebase(
    () => (userId ? doc(firestore, 'users', userId) : null),
    [firestore, userId]
  );
  
  const ordersCollectionRef = useMemoFirebase(
    () => (userId ? collection(firestore, 'users', userId, 'orders') : null),
    [firestore, userId]
  );
  
  const { data: user, isLoading: isUserLoading } = useDoc<User>(userDocRef);
  const { data: orders, isLoading: areOrdersLoading, error } = useCollection<Order>(ordersCollectionRef);

  const isLoading = isUserLoading || areOrdersLoading;

  return (
    <div className="p-4 md:p-8 space-y-6">
       <div className="flex items-center gap-4">
        <Button variant="ghost" size="icon" onClick={() => router.back()}>
            <ArrowLeft className="h-4 w-4" />
        </Button>
        <div>
          <h1 className="text-3xl font-bold">Order History</h1>
          {user ? (
            <p className="text-muted-foreground">Viewing all orders for {user.name}.</p>
          ): (
             <Skeleton className="h-5 w-48 mt-1" />
          )}
        </div>
      </div>

       <Card>
        <CardHeader>
           <CardTitle>All Orders</CardTitle>
           <CardDescription>A list of all orders placed by this customer.</CardDescription>
        </CardHeader>
        <CardContent>
          {isLoading && (
            <div className="space-y-6">
                <Skeleton className="h-48 w-full" />
                <Skeleton className="h-48 w-full" />
            </div>
          )}
           {!isLoading && error && (
            <Alert variant="destructive">
                <ShoppingCart className="h-4 w-4" />
                <AlertTitle>Error Loading Orders</AlertTitle>
                <AlertDescription>
                Could not fetch order data. Please ensure you have the correct permissions.
                </AlertDescription>
            </Alert>
           )}
          {!isLoading && !error && orders && orders.length > 0 && (
            <div className="space-y-6">
                {orders.map((order) => (
                    <OrderCard key={order.id} order={order} />
                ))}
            </div>
          )}
           {!isLoading && !error && (!orders || orders.length === 0) && (
             <Alert>
                <ShoppingCart className="h-4 w-4" />
                <AlertTitle>No Orders Found</AlertTitle>
                <AlertDescription>
                This user has not placed any orders yet.
                </AlertDescription>
            </Alert>
           )}
        </CardContent>
      </Card>
    </div>
  );
}
